/*   The MIT License
*
*   Tempest Engine
*   Copyright (c) 2014 Zdravko Velinov
*
*   Permission is hereby granted, free of charge, to any person obtaining a copy
*   of this software and associated documentation files (the "Software"), to deal
*   in the Software without restriction, including without limitation the rights
*   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*   copies of the Software, and to permit persons to whom the Software is
*   furnished to do so, subject to the following conditions:
*
*   The above copyright notice and this permission notice shall be included in
*   all copies or substantial portions of the Software.
*
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
*   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
*   THE SOFTWARE.
*/

#include "tempest/graphics/opengl-backend/gl-io-command-buffer.hh"
#include "tempest/graphics/opengl-backend/gl-storage.hh"
#include "tempest/graphics/opengl-backend/gl-buffer.hh"
#include "tempest/graphics/opengl-backend/gl-texture.hh"
#include "tempest/graphics/opengl-backend/gl-utils.hh"

namespace Tempest
{
GLIOCommandBuffer::GLIOCommandBuffer(const IOCommandBufferDescription& cmd_desc)
    :   m_IOCommandCount(cmd_desc.CommandCount),
        m_IOCommands(new GLIOCommand[cmd_desc.CommandCount])
{
    glGenFramebuffers(1, &m_FBO);
}

GLIOCommandBuffer::~GLIOCommandBuffer()
{
    glDeleteFramebuffers(1, &m_FBO);
}

void GLIOCommandBuffer::clear()
{
    m_IOCurrentCommand = 0;
}

GLTextureTarget ConvertTo3DTarget(TextureTiling tiling)
{
    switch(tiling)
    {
    default: TGE_ASSERT(false, "Unsupported type");
    case TextureTiling::Array: return GLTextureTarget::GL_TEXTURE_2D_ARRAY;
    case TextureTiling::Volume: return GLTextureTarget::GL_TEXTURE_3D;
    case TextureTiling::Cube: return GLTextureTarget::GL_TEXTURE_CUBE_MAP_ARRAY;
    }
}

GLTextureTarget ConvertTo2DTarget(TextureTiling tiling)
{
    switch(tiling)
    {
    default: TGE_ASSERT(false, "Unsupported type");
    case TextureTiling::Flat: return GLTextureTarget::GL_TEXTURE_2D; break;
    case TextureTiling::Array: return GLTextureTarget::GL_TEXTURE_1D_ARRAY;  break;
    case TextureTiling::Cube: return GLTextureTarget::GL_TEXTURE_CUBE_MAP; break;
    }
}

void GLIOCommandBuffer::_executeCommandBuffer()
{
    for(uint32_t i = 0, iend = m_IOCurrentCommand; i < iend; ++i)
    {
        auto& cmd = m_IOCommands[i];
        switch(cmd.CommandType)
        {
        case IOCommandMode::CopyBuffer:
        {
            cmd.Source.Buffer->bindToTarget(GLBufferTarget::GL_COPY_READ_BUFFER);
            cmd.Destination.Buffer->bindToTarget(GLBufferTarget::GL_COPY_WRITE_BUFFER);
            glCopyBufferSubData(GLBufferTarget::GL_COPY_READ_BUFFER, GLBufferTarget::GL_COPY_WRITE_BUFFER, cmd.SourceOffset, cmd.DestinationOffset, cmd.Width);
        } break;
        case IOCommandMode::CopyTexture:
        {
            auto& src_desc = cmd.Source.Texture->getDescription();
            auto& dst_desc = cmd.Destination.Texture->getDescription();
            glBindFramebuffer(GLFramebufferTarget::GL_READ_FRAMEBUFFER, m_FBO);
            if(dst_desc.Depth > 1)
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width &&
                           cmd.SourceCoordinate.Y + cmd.Height <= src_desc.Height &&
                           cmd.SourceSlice + cmd.Depth <= src_desc.Depth &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width &&
                           cmd.DestinationCoordinate.Y + cmd.Height <= dst_desc.Height &&
                           cmd.DestinationSlice + cmd.Depth <= dst_desc.Depth,
                           "Invalid coordinates specified");
                GLTextureTarget dst_target = ConvertTo3DTarget(dst_desc.Tiling);
                GLTextureTarget src_target = ConvertTo3DTarget(src_desc.Tiling);
                for(uint16_t cur_depth = 0, end_depth = cmd.Depth; cur_depth < end_depth; ++cur_depth)
                {
                    glFramebufferTexture3D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), src_target, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip, cmd.SourceSlice + cur_depth);
#ifndef NDEBUG
                    auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                    TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                    glBindTexture(dst_target, cmd.Destination.Texture->getCPUHandle());
                    glCopyTexSubImage3D(dst_target, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.DestinationCoordinate.Y, cmd.DestinationSlice + cur_depth, cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width, cmd.Height);
                }
            }
            else if(dst_desc.Height > 1)
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width && 
                           cmd.SourceCoordinate.Y + cmd.Height <= src_desc.Height &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width &&
                           cmd.DestinationCoordinate.Y + cmd.Height <= dst_desc.Height,
                           "Invalid coordinates specified");
                GLTextureTarget dst_target = ConvertTo2DTarget(dst_desc.Tiling);
                GLTextureTarget src_target = ConvertTo2DTarget(src_desc.Tiling);
                glFramebufferTexture2D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), src_target, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip);
#ifndef NDEBUG
                auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                glBindTexture(dst_target, cmd.Destination.Texture->getCPUHandle());
                glCopyTexSubImage2D(dst_target, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.DestinationCoordinate.Y, cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width, cmd.Height);
            }
            else
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width,
                           "Invalid coordinates specified");
                glFramebufferTexture1D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), GLTextureTarget::GL_TEXTURE_1D, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip);
#ifndef NDEBUG
                auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                glBindTexture(GLTextureTarget::GL_TEXTURE_1D, cmd.Destination.Texture->getCPUHandle());
                glCopyTexSubImage1D(GLTextureTarget::GL_TEXTURE_1D, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width);
            }
            glBindFramebuffer(GLFramebufferTarget::GL_READ_FRAMEBUFFER, 0);
        } break;
        case IOCommandMode::CopyStorageToBuffer:
        {
            cmd.Source.Storage->bindToTarget(GLBufferTarget::GL_COPY_READ_BUFFER);
            cmd.Destination.Buffer->bindToTarget(GLBufferTarget::GL_COPY_WRITE_BUFFER);
            glCopyBufferSubData(GLBufferTarget::GL_COPY_READ_BUFFER, GLBufferTarget::GL_COPY_WRITE_BUFFER, cmd.SourceOffset, cmd.DestinationOffset, cmd.Width);
        } break;
        case IOCommandMode::CopyStorageToTexture:
        {
            cmd.Source.Storage->bindToTarget(GLBufferTarget::GL_PIXEL_UNPACK_BUFFER);
            auto& dst_desc = cmd.Destination.Texture->getDescription();
            auto line_size = cmd.Width*DataFormatElementSize(dst_desc.Format);

            glPixelStorei(GLPixelStoreMode::GL_UNPACK_ROW_LENGTH, cmd.Width);
            glPixelStorei(GLPixelStoreMode::GL_UNPACK_IMAGE_HEIGHT, cmd.Height);

            auto tex_info = TranslateTextureInfo(dst_desc.Format);

            if(dst_desc.Depth > 1)
            {
                // TODO: check whether it works
                TGE_ASSERT(cmd.SourceOffset + cmd.Height*cmd.Depth*line_size <= cmd.Source.Storage->getSize() &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width &&
                           cmd.DestinationCoordinate.Y + cmd.Height <= dst_desc.Height &&
                           cmd.DestinationSlice + cmd.Depth <= dst_desc.Depth,
                           "Invalid coordinates specified");
                GLTextureTarget target = ConvertTo3DTarget(dst_desc.Tiling);
                glBindTexture(target, cmd.Destination.Texture->getCPUHandle());
                glTexSubImage3D(target, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.DestinationCoordinate.Y, cmd.DestinationSlice, cmd.Width, cmd.Height, cmd.Depth, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.SourceOffset);
            }
            else if(dst_desc.Height > 1)
            {
                TGE_ASSERT(cmd.SourceOffset + cmd.Height*line_size <= cmd.Source.Storage->getSize() &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width &&
                           cmd.DestinationCoordinate.Y + cmd.Height <= dst_desc.Height,
                           "Invalid coordinates specified");
                GLTextureTarget target = ConvertTo2DTarget(dst_desc.Tiling);
                glBindTexture(target, cmd.Destination.Texture->getCPUHandle());
                glTexSubImage2D(target, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.DestinationCoordinate.Y, cmd.Width, cmd.Height, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.SourceOffset);
            }
            else
            {
                TGE_ASSERT(cmd.SourceOffset + line_size <= cmd.Source.Storage->getSize() &&
                           cmd.DestinationCoordinate.X + cmd.Width <= dst_desc.Width,
                           "Invalid coordinates specified");
                glBindTexture(GLTextureTarget::GL_TEXTURE_1D, cmd.Destination.Texture->getCPUHandle());
                glTexSubImage1D(GLTextureTarget::GL_TEXTURE_1D, cmd.DestinationMip, cmd.DestinationCoordinate.X, cmd.Width, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.SourceOffset);
            }
            glBindBuffer(GLBufferTarget::GL_PIXEL_UNPACK_BUFFER, 0);
        } break;
        case IOCommandMode::CopyBufferToStorage:
        {
            cmd.Source.Buffer->bindToTarget(GLBufferTarget::GL_COPY_READ_BUFFER);
            cmd.Destination.Storage->bindToTarget(GLBufferTarget::GL_COPY_WRITE_BUFFER);
            glCopyBufferSubData(GLBufferTarget::GL_COPY_READ_BUFFER, GLBufferTarget::GL_COPY_WRITE_BUFFER, cmd.SourceCoordinate.X, cmd.DestinationCoordinate.X, cmd.Width);
        } break;
        case IOCommandMode::CopyTextureToStorage:
        {
            cmd.Destination.Storage->bindToTarget(GLBufferTarget::GL_PIXEL_PACK_BUFFER);

            glPixelStorei(GLPixelStoreMode::GL_PACK_ROW_LENGTH, cmd.Width);
            glPixelStorei(GLPixelStoreMode::GL_PACK_IMAGE_HEIGHT, cmd.Height);

            auto& src_desc = cmd.Source.Texture->getDescription();
            auto tex_info = TranslateTextureInfo(src_desc.Format);
            auto line_size = cmd.Width*DataFormatElementSize(src_desc.Format);
            glBindFramebuffer(GLFramebufferTarget::GL_READ_FRAMEBUFFER, m_FBO);
            if(src_desc.Depth > 1)
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width &&
                           cmd.SourceCoordinate.Y + cmd.Height <= src_desc.Height &&
                           cmd.SourceSlice + cmd.Depth <= src_desc.Depth &&
                           cmd.DestinationOffset + cmd.Height*cmd.Depth*line_size <= cmd.Destination.Storage->getSize(),
                           "Invalid coordinates specified");
                GLTextureTarget target = ConvertTo3DTarget(src_desc.Tiling);
                for(uint16_t cur_depth = 0, end_depth = cmd.Depth; cur_depth < end_depth; ++cur_depth)
                {
                    glFramebufferTexture3D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), target, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip, cmd.SourceSlice + cur_depth);
                    glReadBuffer(UINT_TO_GL_BUFFER_COLOR_ATTACHMENT(0));
#ifndef NDEBUG
                    auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                    TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                    glReadPixels(cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width, cmd.Height, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.DestinationOffset + cur_depth*cmd.Height*cur_depth);
                }
            }
            else if(src_desc.Height > 1)
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width &&
                           cmd.SourceCoordinate.Y + cmd.Height <= src_desc.Height &&
                           cmd.DestinationOffset + cmd.Height*line_size <= cmd.Destination.Storage->getSize(),
                           "Invalid coordinates specified");
                GLTextureTarget target = ConvertTo2DTarget(src_desc.Tiling);
                glFramebufferTexture2D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), target, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip);
                glReadBuffer(UINT_TO_GL_BUFFER_COLOR_ATTACHMENT(0));
#ifndef NDEBUG
                auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                glReadPixels(cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width, cmd.Height, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.DestinationOffset);
                CheckOpenGL();
            }
            else
            {
                TGE_ASSERT(cmd.SourceCoordinate.X + cmd.Width <= src_desc.Width &&
                           cmd.DestinationOffset + line_size <= cmd.Destination.Storage->getSize(),
                           "Invalid coordinates specified");
                glFramebufferTexture1D(GLFramebufferTarget::GL_READ_FRAMEBUFFER, UINT_TO_GL_COLOR_ATTACHMENT(0), GLTextureTarget::GL_TEXTURE_1D, cmd.Source.Texture->getCPUHandle(), cmd.SourceMip);
                glReadBuffer(UINT_TO_GL_BUFFER_COLOR_ATTACHMENT(0));
#ifndef NDEBUG
                auto status = glCheckFramebufferStatus(GLFramebufferTarget::GL_READ_FRAMEBUFFER);
                TGE_ASSERT(status == GLFramebufferStatus::GL_FRAMEBUFFER_COMPLETE, "Framebuffer is broken");
#endif
                glReadPixels(cmd.SourceCoordinate.X, cmd.SourceCoordinate.Y, cmd.Width, cmd.Height, tex_info.Format, tex_info.Type, static_cast<char*>(nullptr) + cmd.DestinationOffset);
            }
            glBindFramebuffer(GLFramebufferTarget::GL_READ_FRAMEBUFFER, 0);
            glBindBuffer(GLBufferTarget::GL_PIXEL_PACK_BUFFER, 0);
        } break;
        default: TGE_ASSERT(false, "Unsupported command type");
        }
        CheckOpenGL();
    }
}
}